  -- Imnoo machineKey zu einer Kostenstelle finden
    CREATE OR REPLACE FUNCTION x_10_interfaces.imnoo__machineKey__by_ks_abt__get(
      IN  _ks_abt       varchar,
      OUT _machineKey  varchar
      )
      RETURNS varchar AS $$
      BEGIN

        SELECT imi_parameter_value
          INTO _machineKey
          FROM x_10_interfaces.interfaces_mapping_id
        WHERE imi_connector = 'imnoo-API'
          AND imi_table = 'ksv'::regclass
          AND imi_column = 'ks_abt'
          AND imi_value = _ks_abt
          AND imi_parameter_id = 'machineKey'
        ORDER BY imi_id DESC
        LIMIT 1;

      END $$ LANGUAGE plpgsql;
  --

  -- Imnoo articleKey zu einem Artikel finden
    CREATE OR REPLACE FUNCTION x_10_interfaces.imnoo__articleKey__by_ak_nr__get(
      IN  _ak_nr       varchar,
      OUT _articleKey  varchar
      )
      RETURNS varchar AS $$
      BEGIN

        SELECT imi_parameter_value
          INTO _articleKey
          FROM x_10_interfaces.interfaces_mapping_id
        WHERE imi_connector = 'imnoo-API'
          AND imi_table = 'art'::regclass
          AND imi_column = 'ak_nr'
          AND imi_value = _ak_nr
          AND imi_parameter_id = 'articleKey'
        ORDER BY imi_id DESC
        LIMIT 1;

      END $$ LANGUAGE plpgsql;
  --

  -- Imnoo production_run zu einer ABK finden
    CREATE OR REPLACE FUNCTION x_10_interfaces.imnoo__production_run__by_ab_ix__get(
      IN  _ab_ix           integer,
      OUT _production_run  varchar
      )
      RETURNS varchar AS $$
      BEGIN

        SELECT imi_parameter_value
          INTO _production_run
          FROM x_10_interfaces.interfaces_mapping_id
        WHERE imi_connector = 'imnoo-API'
          AND imi_table = 'abk'::regclass
          AND imi_column = 'ab_ix'
          AND imi_value = _ab_ix
          AND imi_parameter_id = 'productionRunId'
        ORDER BY imi_id DESC
        LIMIT 1;

      END $$ LANGUAGE plpgsql;
  --


  -- Imnoo machineKey zu einer Kostenstelle definieren
    CREATE OR REPLACE FUNCTION x_10_interfaces.imnoo__machineKey__for_ks_abt__set(
      IN  _ks_abt      varchar,
      IN  _machineKey  varchar
      )
      RETURNS void AS $$
      BEGIN

        -- bereits vorhandene imnoo-Maschinen zu einer Kostenstelle löschen
        DELETE FROM x_10_interfaces.interfaces_mapping_id
              WHERE imi_connector = 'imnoo-API'
                AND imi_table = 'ksv'::regclass
                AND imi_column = 'ks_abt'
                AND imi_value = _ks_abt
                AND imi_parameter_id = 'machineKey';

        -- Kostenstellenzuordnung schreiben
        INSERT INTO x_10_interfaces.interfaces_mapping_id ( imi_connector, imi_table,        imi_column, imi_value, imi_parameter_id, imi_parameter_value )
                    VALUES                                ( 'imnoo-API',   'ksv'::regclass, 'ks_abt',    _ks_abt,   'machineKey',     _machineKey         );

      END $$ LANGUAGE plpgsql;
  --


  -- Liste der Auftragszeiten je Kostenstelle zu einer ABK erhalten
    CREATE OR REPLACE FUNCTION x_10_interfaces.zeiten_je_kostenstelle__by_ab_ix__get(
      IN  _ab_ix           integer
      )
      RETURNS TABLE (ks_abt varchar, _machinekey varchar, th numeric, tr numeric) AS $$
      BEGIN

      -- Arbeitsgänge mit Zeiten ermitteln und anschließend
      -- Arbeitsgänge mit vorhandener machineKey-Zuordnung als JSON-Array gemäß imnoo-Notation aufbauen
      RETURN QUERY
      WITH kostenstellen AS (
        SELECT DISTINCT ba_ks AS kost, _ab_ix FROM bdea WHERE ba_ix = _ab_ix
          UNION
        SELECT DISTINCT  r_ks AS kost, _ab_ix FROM rm   WHERE r_a2_id IN (SELECT a2_id FROM ab2 WHERE a2_ab_ix = _ab_ix)
          UNION
        SELECT DISTINCT n2_ks AS kost, _ab_ix FROM nk2  WHERE n2_ix = _ab_ix)
      SELECT kost AS ks_abt,
             keys._machinekey,
             coalesce(nachkalkulation.th, rueckmeldungen.th, stempelungen.th) / ab_st_uf1 / 60 AS th,
             coalesce(nachkalkulation.tr, rueckmeldungen.tr, stempelungen.tr) / 3600 AS tr
        FROM kostenstellen
        JOIN abk on kostenstellen._ab_ix = ab_ix
        -- Stempelungen aus BDE
        LEFT JOIN LATERAL (SELECT ba_ks,
                                  sum(CASE WHEN ba_ruest THEN ba_efftime ELSE 0 END) as tr,
                                  sum(CASE WHEN NOT ba_ruest THEN ba_efftime ELSE 0 END) as th
                            FROM bdea
                            WHERE ba_ix = ab_ix
                            GROUP BY ba_ks) AS stempelungen ON ba_ks = kost
        -- RÜckmeldungen
        LEFT JOIN LATERAL (SELECT r_ks,
                                  sum(CASE WHEN r_ruest THEN r_std_sek ELSE 0 END) as tr,
                                  sum(CASE WHEN NOT r_ruest THEN r_std_sek ELSE 0 END) as th
                            FROM rm
                            WHERE r_a2_id IN (SELECT a2_id FROM ab2 WHERE a2_ab_ix = ab_ix)
                            GROUP BY r_ks) AS rueckmeldungen ON r_ks = kost
        --Nachkalkulationsdaten
        LEFT JOIN LATERAL (SELECT n2_ks,
                                  sum(CASE WHEN n2_ruest AND NOT n2_ausw THEN n2_ez_stu ELSE 0 END) * 3600 as tr,
                                  sum(CASE WHEN NOT n2_ruest AND NOT n2_ausw THEN n2_ez_stu ELSE 0 END) * 3600 as th
                            FROM nk2
                            WHERE n2_ix = ab_ix
                            GROUP BY n2_ks) AS nachkalkulation ON n2_ks = kost
        -- imnoo machineKey-Zuordnung ergänzen
        LEFT JOIN x_10_interfaces.imnoo__machineKey__by_ks_abt__get(kost) AS keys ON true;

      END $$ LANGUAGE plpgsql;
  --


  -- Produktionslauf-Daten aus ABK für imnoo zusammenstellen
    CREATE OR REPLACE FUNCTION x_10_interfaces.imnoo__production_run_jsonb__by_ab_ix__get(
      IN  _ab_ix           integer,
      OUT _production_run  jsonb
      )
      RETURNS jsonb AS $$
      DECLARE
        manufacturing_steps jsonb;
      BEGIN

      -- Arbeitsgänge mit Zeiten ermitteln und anschließend
      -- Arbeitsgänge mit vorhandener machineKey-Zuordnung als JSON-Array gemäß imnoo-Notation aufbauen
      SELECT jsonb_agg(
              jsonb_build_object(
                'machineKey', _machinekey,
                'costUnits', jsonb_build_object(
                  'machining', jsonb_build_object('time', greatest( th * 60, 1)), -- Laufzeit muss >0 sein
                  'setup', jsonb_build_object('time', tr * 3600),
                  'programming', jsonb_build_object('time', 0)
                )
              )
            )
        INTO manufacturing_steps
        FROM x_10_interfaces.zeiten_je_kostenstelle__by_ab_ix__get(_ab_ix) AS kostenstellen
        JOIN abk ON _ab_ix = ab_ix
      WHERE _machinekey IS NOT NULL;

      -- JSON zum Produktionslauf erstellen
      _production_run = (SELECT jsonb_build_object(
                                'batchNumber', ab_ix::varchar,
                                'quantity', ab_st_uf1,
                                'productionDate', to_char(ab_et::timestamp with time zone, 'YYYY-MM-DD"T"HH24:MI:SS"Z"'),
                                'manufacturingSteps', manufacturing_steps
                              )
                          FROM abk
                        WHERE ab_ix = _ab_ix);

      END $$ LANGUAGE plpgsql;
  --


  -- Liste der Auftragszeiten je Kostenstelle zu einer ASK erhalten
    CREATE OR REPLACE FUNCTION x_10_interfaces.zeiten_je_kostenstelle__by_op_ix__get(
        IN  _op_ix           integer
      )
      RETURNS TABLE (ks_abt varchar, _machinekey varchar, th numeric, tr numeric) AS $$
      BEGIN

      -- Arbeitsgänge mit Zeiten ermitteln und anschließend
      -- Arbeitsgänge mit vorhandener machineKey-Zuordnung als JSON-Array gemäß imnoo-Notation aufbauen
      RETURN QUERY
      SELECT o2_ks,
             keys._machinekey,
             o2_th_sek / 60 AS th,
             o2_tr_sek / 3600 AS tr
        FROM opl
        JOIN op2 ON o2_ix = op_ix
        JOIN x_10_interfaces.imnoo__machineKey__by_ks_abt__get(o2_ks) AS keys ON true
       WHERE op_ix = _op_ix
         AND keys._machinekey IS NOT NULL;

      END $$ LANGUAGE plpgsql;
  --


  -- Produktionslauf-Daten aus ASK für imnoo zusammenstellen
    CREATE OR REPLACE FUNCTION x_10_interfaces.imnoo__production_run_jsonb__by_op_ix__get(
        IN  _op_ix           integer,
        OUT _production_run  jsonb
      )
      RETURNS jsonb AS $$
      DECLARE
        manufacturing_steps jsonb;
      BEGIN

      -- Arbeitsgänge mit Zeiten ermitteln und anschließend
      -- Arbeitsgänge mit vorhandener machineKey-Zuordnung als JSON-Array gemäß imnoo-Notation aufbauen
      SELECT jsonb_agg(
              jsonb_build_object(
                'machineKey', _machinekey,
                'costUnits', jsonb_build_object(
                  'machining', jsonb_build_object('time', greatest( th * 60, 1)), -- Laufzeit muss >0 sein
                  'setup', jsonb_build_object('time', tr * 3600),
                  'programming', jsonb_build_object('time', 0)
                )
              )
             )
        INTO manufacturing_steps
        FROM x_10_interfaces.zeiten_je_kostenstelle__by_op_ix__get(_op_ix) AS kostenstellen;

      -- JSON zum Produktionslauf erstellen
      _production_run = (SELECT jsonb_build_object(
                                  'batchNumber', 'ASK-' || op_ix::varchar,
                                  'quantity', op_lg,
                                  'productionDate', to_char(opl.insert_date::timestamp with time zone, 'YYYY-MM-DD"T"HH24:MI:SS"Z"'),
                                  'manufacturingSteps', manufacturing_steps
                                )
                           FROM opl
                          WHERE op_ix = _op_ix);

      END $$ LANGUAGE plpgsql;
  --